﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common.Extensions;
using VA.PPMS.IWS.MappingService.Helpers;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapContactToCrm : MapperRelatedBase
    {
        public MapContactToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
            : base(ppmsContextHelper, ppmsHelper)
        {}

        public async Task<Contact> MapInsert(PersonContact contact, Account provider)
        {
            // map entity
            var entity = new Contact
            {
                Id = Guid.NewGuid(),
                FirstName = contact.FirstName,
                MiddleName = contact.MiddleName,
                LastName = contact.LastName
            };

            if (contact.IsVeteranSpecified)
            {
                entity.ppms_isveteran = contact.IsVeteran;
            }

            if (contact.RoleSpecified)
            {
                entity.AccountRoleCode = EnumHelper.MapEnumToOptionSetValue<Contact_AccountRoleCode>(contact.Role.ToString());
            }

            if (contact.EffectiveDateSpecified && contact.EffectiveDate.Year > EnumHelper.MinYear)
            {
                entity.ppms_effectivedate = contact.EffectiveDate.ToCrmDate();
            }

            // Emails
            if (contact.Emails != null && contact.Emails.Item != null && contact.Emails.Item.Any())
            {
                int i = 1;
                foreach (var item in contact.Emails.Item)
                {
                    switch (i++)
                    {
                        case 1:
                            entity.EMailAddress1 = item.EmailAddress;
                            break;
                        case 2:
                            entity.EMailAddress2 = item.EmailAddress;
                            break;
                        case 3:
                            entity.EMailAddress3 = item.EmailAddress;
                            break;
                        default:
                            break;
                    }
                }
            }

            // Addresses
            if (contact.Addresses != null && contact.Addresses.Item != null && contact.Addresses.Item.Any())
            {
                int i = 1;
                foreach (var item in contact.Addresses.Item)
                {
                    switch (i++)
                    {
                        case 1:
                            entity.Address1_Line1 = item.Address1;
                            entity.Address1_Line2 = item.Address2;
                            entity.Address1_Line3 = item.Address3;
                            entity.Address1_City = item.City;
                            entity.Address1_StateOrProvince = item.State;
                            entity.Address1_PostalCode = item.PostalCode;
                            entity.Address1_County = item.County;
                            entity.Address1_Country = item.CountryCode;
                            break;
                        case 2:
                            entity.Address2_Line1 = item.Address1;
                            entity.Address2_Line2 = item.Address2;
                            entity.Address2_Line3 = item.Address3;
                            entity.Address2_City = item.City;
                            entity.Address2_StateOrProvince = item.State;
                            entity.Address2_PostalCode = item.PostalCode;
                            entity.Address2_County = item.County;
                            entity.Address2_Country = item.CountryCode;
                            break;
                        case 3:
                            entity.Address3_Line1 = item.Address1;
                            entity.Address3_Line2 = item.Address2;
                            entity.Address3_Line3 = item.Address3;
                            entity.Address3_City = item.City;
                            entity.Address3_StateOrProvince = item.State;
                            entity.Address3_PostalCode = item.PostalCode;
                            entity.Address3_County = item.County;
                            entity.Address3_Country = item.CountryCode;
                            break;
                        default:
                            break;
                    }
                }
            }

            // Phones
            if (contact.Phones != null && contact.Phones.Item != null && contact.Phones.Item.Any())
            {
                int i = 1;
                foreach (var item in contact.Phones.Item)
                {
                    switch(i++)
                    {
                        case 1:
                            entity.Telephone1 = item.PhoneNumber;
                            entity.ppms_istextingacceptable = item.IsTextingAcceptableSpecified ? item.IsTextingAcceptable : false;
                            break;
                        case 2:
                            entity.Telephone2 = item.PhoneNumber;
                            break;
                        case 3:
                            entity.Telephone3 = item.PhoneNumber;
                            break;
                        default:
                            break;
                    }
                }
            }

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) entity.OwnerId = Owner;

            await Task.Run(() => {});

            return entity;
        }

        public async Task<Contact> MapUpdate(PersonContact entity, Account provider)
        {
            Contact crmEntity = null;

            if (string.IsNullOrEmpty(entity.CorrelationId))
            {
                // Get related entities with matching name
                crmEntity = GetContactByName(provider, entity.FirstName, entity.MiddleName, entity.LastName);
            }
            else
            {
                // get entity, if CorrelationId provided
                crmEntity = GetCrmEntity(provider, entity.CorrelationId);
            }

            // matching account not found
            if (crmEntity == null) return await MapInsert(entity, provider);

            var newEntity = new Contact()
            {
                Id = crmEntity.Id,
                FirstName = crmEntity.FirstName,
                MiddleName = crmEntity.MiddleName,
                LastName = crmEntity.LastName,
                ppms_effectivedate = crmEntity.ppms_effectivedate
            };

            // Map fields
            if (IsChanged(entity.FirstName, newEntity.FirstName))
                newEntity.FirstName = entity.FirstName;

            if (IsChanged(entity.MiddleName, newEntity.MiddleName))
                newEntity.MiddleName = entity.MiddleName;

            if (IsChanged(entity.LastName, newEntity.LastName))
                newEntity.LastName = entity.LastName;

            if (entity.IsVeteranSpecified)
                newEntity.ppms_isveteran = entity.IsVeteran;

            if (entity.RoleSpecified)
                newEntity.AccountRoleCode = EnumHelper.MapEnumToOptionSetValue<Contact_AccountRoleCode>(entity.Role.ToString());

            if (entity.EffectiveDateSpecified && entity.EffectiveDate.Year > EnumHelper.MinYear)
                newEntity.ppms_effectivedate = entity.EffectiveDate.ToCrmDate();

            // Emails
            if (entity.Emails != null && entity.Emails.Item != null && entity.Emails.Item.Any())
            {
                int i = 1;
                foreach (var item in entity.Emails.Item)
                {
                    switch (i++)
                    {
                        case 1:
                            newEntity.EMailAddress1 = item.EmailAddress;
                            break;
                        case 2:
                            newEntity.EMailAddress2 = item.EmailAddress;
                            break;
                        case 3:
                            newEntity.EMailAddress3 = item.EmailAddress;
                            break;
                        default:
                            break;
                    }
                }
            }

            // Addresses
            if (entity.Addresses != null && entity.Addresses.Item != null && entity.Addresses.Item.Any())
            {
                int i = 1;
                foreach (var item in entity.Addresses.Item)
                {
                    switch (i++)
                    {
                        case 1:
                            newEntity.Address1_Line1 = item.Address1;
                            newEntity.Address1_Line2 = item.Address2;
                            newEntity.Address1_Line3 = item.Address3;
                            newEntity.Address1_City = item.City;
                            newEntity.Address1_StateOrProvince = item.State;
                            newEntity.Address1_PostalCode = item.PostalCode;
                            newEntity.Address1_County = item.County;
                            newEntity.Address1_Country = item.CountryCode;
                            break;
                        case 2:
                            newEntity.Address2_Line1 = item.Address1;
                            newEntity.Address2_Line2 = item.Address2;
                            newEntity.Address2_Line3 = item.Address3;
                            newEntity.Address2_City = item.City;
                            newEntity.Address2_StateOrProvince = item.State;
                            newEntity.Address2_PostalCode = item.PostalCode;
                            newEntity.Address2_County = item.County;
                            newEntity.Address2_Country = item.CountryCode;
                            break;
                        case 3:
                            newEntity.Address3_Line1 = item.Address1;
                            newEntity.Address3_Line2 = item.Address2;
                            newEntity.Address3_Line3 = item.Address3;
                            newEntity.Address3_City = item.City;
                            newEntity.Address3_StateOrProvince = item.State;
                            newEntity.Address3_PostalCode = item.PostalCode;
                            newEntity.Address3_County = item.County;
                            newEntity.Address3_Country = item.CountryCode;
                            break;
                        default:
                            break;
                    }
                }
            }

            // Phones
            if (entity.Phones != null && entity.Phones.Item != null && entity.Phones.Item.Any())
            {
                int i = 1;
                foreach (var item in entity.Phones.Item)
                {
                    switch (i++)
                    {
                        case 1:
                            newEntity.Telephone1 = item.PhoneNumber;
                            newEntity.ppms_istextingacceptable = item.IsTextingAcceptableSpecified ? item.IsTextingAcceptable : false;
                            break;
                        case 2:
                            newEntity.Telephone2 = item.PhoneNumber;
                            break;
                        case 3:
                            newEntity.Telephone3 = item.PhoneNumber;
                            break;
                        default:
                            break;
                    }
                }
            }

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) newEntity.OwnerId = Owner;

            return newEntity;
        }

        public SetStateRequest MapDelete(Contact crmItem)
        {
            if (crmItem != null)
            {
                return DeactivateEntity(crmItem);
            }
            return null;
        }

        private Contact GetCrmEntity(Account provider, string id)
        {
            var list = provider.contact_customer_accounts;
            if (list == null) return null;

            var guid = new Guid(id);
            var contacts = list as Contact[] ?? list.ToArray();

            return contacts.Any() ? contacts.FirstOrDefault(x => x.Id == guid) : null;
        }

        private Contact GetContactByName(Account provider, string firstName, string middleName, string lastName)
        {
            Account parent = ParentalReference != null ? (Account)ParentalReference : provider;
            if (parent == null) parent = provider;

            var list = parent.contact_customer_accounts;
            if (list == null) return null;

            var contacts = list as Contact[] ?? list.ToArray();

            return contacts.Any() ? contacts.FirstOrDefault(x => x.FirstName == firstName && x.LastName == lastName && x.MiddleName == middleName) : null;
        }

        private static PersonContact ConvertEntity<T>(T entity)
        {
            return (PersonContact)Convert.ChangeType(entity, typeof(PersonContact));
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (IsWithinContext) AssociateRelatedEntities(parent, entities, "contact_customer_accounts");
            else
            {
                if (entities != null && entities.Count > 0)
                {
                    var account = (Account)parent;
                    if (account != null)
                        account.contact_customer_accounts = ConvertEntityList<Contact>(entities);
                }
            }
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<Contact>();
            var relatedEntities = provider.contact_customer_accounts.ToList();
            PersonContact entity;

            if (relatedEntities != null && relatedEntities.Any())
            {
                // Map schema entities for delete
                foreach (var item in entities)
                {
                    entity = ConvertEntity(item);
                    var matches = provider.contact_customer_accounts.Where(p => p.Id == new Guid(entity.CorrelationId));
                    list.AddRange(matches);
                }
            }

            return EntityDelete((IEnumerable<Contact>)list);
        }
    }
}